# distinct_diagonal_check.py
#
# Usage: python distinct_diagonal_check.py (size of G)
# 
# This code will determine if all partial transversals of length 3
# in C(G), where G = Z_25, Z_35, or Z_49, for which (0,0,0) is a cell
# and the cells belong to distinct block diagonals with respect to H,
# is completable to a transversal of C(G). The only permissible inputs
# for the sizes of G are 25, 35, and 49.

import random
import sys

from itertools import permutations
from itertools import combinations

MAX_RUN=50
SUCCEEDED=1
FAILED=0

if( len(sys.argv) > 1 ):
	modulus = int(sys.argv[1])
else:
	modulus = 25

if( modulus == 25 ):
	mm = 5
	qq = 5
elif( modulus == 35):
	mm = 7
	qq = 5
elif( modulus == 49):
	mm = 7
	qq = 7

# The array bad(n)cases consists of all partial transversals in C(G)
# of length 3 which are also contained in C(H) and are not completable
# to a transversal in C(H).

bad25cases = [
{(5,5),(20,10)},
{(15,5),(20,10)},
{(10,5),(15,15)},
{(10,5),(20,15)},
{(5,5),(10,20)},
{(15,5),(20,20)},
{(5,10),(15,15)},
{(10,10),(20,15)},
{(5,10),(15,20)},
{(10,10),(15,20)},
{(5,15),(10,20)},
{(5,15),(20,20)}]

bad49cases = [
{(7,7),(42,14)},
{(35,7),(42,14)},
{(21,7),(28,28)},
{(21,7),(42,28)},
{(7,7),(14,42)},
{(35,7),(42,42)},
{(14,14),(35,28)},
{(21,14),(35,28)},
{(14,14),(28,35)},
{(21,14),(35,35)},
{(7,21),(28,28)},
{(21,21),(42,28)},
{(14,21),(28,35)},
{(14,21),(35,35)},
{(7,21),(28,42)},
{(21,21),(28,42)},
{(7,35),(14,42)},
{(7,35),(42,42)}]

bad35cases = [
{(5,5),(30,10)},
{(25,5),(30,10)},
{(15,5),(20,20)},
{(15,5),(30,20)},
{(5,5),(10,30)},
{(25,5),(30,30)},
{(10,10),(25,20)},
{(15,10),(25,20)},
{(10,10),(20,25)},
{(15,10),(25,25)},
{(5,15),(20,20)},
{(15,15),(30,20)},
{(10,15),(20,25)},
{(10,15),(25,25)},
{(5,15),(20,30)},
{(15,15),(20,30)},
{(5,25),(10,30)},
{(5,25),(30,30)}]








# Routine: dualChainCreate
# input:  a  - the row of c_1
#         b  - the column of c_1
#         aa - the row of c_2
#         bb - the column of c_2
# output: x  - the rows of cells in a chain whose swap contains c_1
#         y  - the columns of cells in a chain whose swap contains c_1
#         d  - the diagonal for which zip(x,y) is a chain
#         xx - the rows of cells in a chain whose swap contains c_2
#         yy - the columns of cells in a chain whose swap contains c_2
#         e  - the diagonal for which zip(xx,yy) is a chain

# This routine takes in a pair of cells which are off the main diagonal,
# and it is assumed they below to different block diagonals with respect to H.
# It first attempts to find a (d,H)-chain whose swap contains c_1 and avoids
# (0,0,0) and c_2. Then it attempts to find an (e,H)-chain whose swap contains c_2
# which avoids both (0,0,0) and the first chain.

# In using the chains to find a transversal, it is necessary that the cells 
# in the chain contained in C(H), along with (0,0,0), must be completable to 
# a transversal in C(H). This routine checks to see if the chains found produce
# any non-completable partial transversal of this type in C(H). If so, the routine
# will try to find another pair of disjoint chains.

def dualChainCreate(a,b,aa,bb):
	d = (a-b)%qq
	e = (aa-bb)%qq
	counter = 0
	while(counter < MAX_RUN ):
	
		counter = counter + 1
		
		w,z,result = cellConflictAvoidRandom(a,b,[0,aa],[0,bb])
		if( result == FAILED ): continue
		
		x,y,result = permissibleChainCreate(a,b,w,z,[0,(aa+bb)%modulus])
		if( result == FAILED ): continue
		
		x,y,result = augmentChainRandom(x,y,d,[0,aa],[0,bb],[0,(aa+bb)%modulus])
		if( result == FAILED ): continue
		
		usedSymbols = [(i+j)%modulus for i,j in zip(x,y)]

		ww,zz,result = cellConflictAvoidRandom(aa,bb,[0]+x,[0]+y)
		if( result == FAILED ): continue

		xx,yy,result = permissibleChainCreate(aa,bb,ww,zz,[0]+usedSymbols)
		if( result == FAILED ): continue

		xx,yy,result = augmentChainRandom(xx,yy,e,[0]+x,[0]+y,[0]+usedSymbols)
		if( result == FAILED ): continue

		if( mm == 5 and qq == 5 and not( {(x[0],y[0]),(xx[0],yy[0])} in bad25cases ) ):
			return x,y,d,xx,yy,e,SUCCEEDED

		if( mm == 7 and qq == 5 and not( {(x[0],y[0]),(xx[0],yy[0])} in bad35cases ) ):
			return x,y,d,xx,yy,e,SUCCEEDED

		if( mm == 7 and qq == 7 and not( {(x[0],y[0]),(xx[0],yy[0])} in bad49cases ) ):
			return x,y,d,xx,yy,e,SUCCEEDED
			
	return x,y,d,xx,yy,e,FAILED







# Routine: cellConflictAvoidRandom
# input:  a -          the row of c_i
#         b -          the column of c_i
#         badrows -    rows which must be avoided by the chain
#         badcolumns - columns which must be avoided by the chain
# output: the row    for a cell which contains a+b on the main block diagonal
#         the column for a cell which contains a+b on the main block diagonal

# In finding a chain whose swap contains a particular cell, we must first find 
# a cell on the main block diagonal which shares the cell's symbol. This routine
# outputs the row and column of a cell on the main block diagonal which matches 
# the symbol a+b whose row and column are not listed in badrows or badcolumns.

def cellConflictAvoidRandom(a,b,badrows,badcolumns):
	# This routine will identify a cell in the main block diagonal
	# which does not conflict with the rows and columns provided.
	k = (qq - 1) // 2
	s = (a + b) % modulus
	c = ((k+1)*s) % qq
	counter = 0
	while( counter < MAX_RUN ):
		counter = counter + 1
		possibleRow = c + random.randint(0, mm-1)*qq
		if( not( possibleRow in badrows or (s-possibleRow)%modulus in badcolumns ) ):
			return possibleRow, (s-possibleRow)%modulus, SUCCEEDED
	return 0,0,FAILED







# Routine: augmentChainRandom
# input:  x -          the rows for the cells in a chain
#         y -          the columns for the cells in a chain
#         d -          the diagonal index for the chain
#         badrows -    rows which must be avoided by the chain
#         badcolumns - columns which must be avoided by the chain
#         badsymbols - symbols which must be avoided by the chain
# output: rows for cells in an augmented chain
#         columns for cells in an augmented chain

# This routine will take in a (d,H)-chain as an input, and this chain
# will not have any conflict with badsymbols, but may have an overlap
# with badrows or badcolumns. For each conflict, this routine will
# augment the chain by chaining one row and one column [and therefore 
# two symbols] in an attempt to remove the conflicting cells.

def augmentChainRandom(x,y,d,badrows,badcolumns,badsymbols):
	k = (qq - 1) // 2
	for i in range(qq):
		if x[(i*d)%qq] in badrows or y[(d*(i+k))%qq] in badcolumns:
			corrected = 0
			counter = 0
			while( corrected == 0 and counter < MAX_RUN ):
				counter = counter + 1
				possibleShift = random.randint(1,mm-1)*qq
				if( not( (x[(i*d)%qq]+possibleShift)%modulus in badrows or
				         (x[(i*d)%qq]+possibleShift+y[(i*d)%qq])%modulus in badsymbols or
				         (x[(d*(i+k))%qq]+possibleShift+y[(d*(i+k))%qq])%modulus in badsymbols or
				         (y[(d*(i+k))%qq]+possibleShift)%modulus in badcolumns
					    )
				  ):
					x[(i*d)%qq] = (x[(i*d)%qq]+possibleShift)%modulus
					y[(d*(i+k))%qq] = (y[(d*(i+k))%qq]+possibleShift)%modulus 
					corrected = 1
			if( corrected == 0 ):
				return [],[],FAILED
	return x,y,SUCCEEDED







# Routine: findSymbolsRandom
# input:  necessarySymbols - a set of symbols which must be used by an SDR of G/H
#         badSymbols -       a set of symbols which must be avoided in an SDR of G/H
# output: symbolValues - an SDR of symbols from G/H, if successful

# This routine will, for each coset of H, attempt to find a representative symbol.
# If there already is a symbol in necessarySymbols which belongs to the coset, that symbol
# is included in the SDR. Otherwise, this routine will randomly generate an element of 
# the coset, then check to see if it belongs to the set of badSymbols. If the randomly 
# generated symbol does not belong to badSymbols, it is kept for the SDR; otherwise another
# symbol is randomly generated.

def findSymbolsRandom(necessarySymbols,badSymbols,mma,qqa):
	symbolValues = [0] * qq
	necessaryAlreadySelected = necessarySymbols
	counter = 0
	for j in necessarySymbols:
		symbolValues[j%qq] = j
		necessaryAlreadySelected[counter] = j%qq
		counter = counter + 1
	for i in range(qq):
		if( i in necessaryAlreadySelected ):
			continue
		counter = 0
		acceptable = 0
		while( counter < MAX_RUN and acceptable == 0 ):
			counter = counter + 1 
			possibleSymbol = i + random.randint(0, mm-1)*qq
			if( not( possibleSymbol in badSymbols ) ):
				symbolValues[i] = possibleSymbol
				acceptable = 1
		if( acceptable == 0 ):
			return [],FAILED
	return symbolValues, SUCCEEDED







# Routine: permissibleChainCreate
# input:  a -          the row    of c_i
#         b -          the column of c_i
#         w -          the row    of a cell with the symbol of c_i on the main block diagonal
#         z -          the column of a cell with the symbol of c_i on the main block diagonal
#         badSymbols - a set of symbols which must be avoided in selecting those for a chain
# output: the rows of cells in a chain with these specified requirements
#         the columns of cells in a chain with these specified requirements

# When building a chain with respect to a cell c_i that avoids a set of excluded symbols,
# we must first find an SDR which uses the symbol of c_i which does not intersect badSymbols.
# This code randomly selects an SDR which avoids badSymbols, while making sure the symbol selection
# will correspond to a (d,H)-chain that contains (w,z,a+b) and whose swap contains (a,b,a+b),
# all the while avoiding symbols in the set badSymbols.

def permissibleChainCreate(a,b,w,z,badSymbols):
	start = (w+z)%qq
	astart = w%qq
	k = (qq - 1) // 2
	d = (a-b)%qq

	succeeded = 0
	counter = 0
	while( succeeded == 0 and counter < MAX_RUN):
		counter = counter + 1
		symbolValues,result = findSymbolsRandom([w+z],badSymbols,mm,qq)
		if( result == FAILED ):
			return [],[],FAILED
		# Once the first run of symbols are created, the last one
		# must be determined to hit the desired cell (w,z)
		index = (2*k*d+start)%qq
		symbolValues[index] = (w+b)%(mm*qq)

		for i in range(2*k):
			symbolValues[index] = (symbolValues[index] - (-1)**i*symbolValues[(d*i+start)%qq])%(mm*qq)
		if( not( symbolValues[index] in badSymbols ) ):
			succeeded = 1
	if( succeeded == 1 ):
		return chainCreate(w,symbolValues,d)
	else:
		return [],[],FAILED







# Routine: chainCreate
# input:  a -            a row used by the chain to be created
#         symbolValues - an SDR of G/H corresponding to the symbols to be used by the chain
#         d -            the index of the block diagonal for the chain
# output: x - the rows for a chain with specified symbol values and row
#         y - the column for a chain with specified symbol values and row

# A (d,H)-chain can be determined uniquely by the SDR of G/H corresponding to its symbol set
# as well as one row (or column) specified. In this routine, a chain is created given a row
# and the set of symbols to be used by the chain.

def chainCreate(a,symbolValues,d):
# will produce a collection of 2q cells which form a chain
# uses row a and the q symbols in symbolValues and diagonal d
# x and y are indexed by the symbol set from which the rows and columns arise
# the value of d is needed to order this as a chain.
	astart = a%qq
	k = (qq - 1) // 2
	x = [0] * qq
	y = [0] * qq
	x[astart] = a
	y[astart] = symbolValues[(astart*2)%qq] - x[astart]
	for i in range(qq):
		x[(astart+(i+1)*d)%qq] = (symbolValues[(2*astart+(2*(i+1)-1)*d)%qq] - y[(astart+i*d)%qq])%modulus
		y[(astart+(i+1)*d)%qq] = (symbolValues[(2*astart+2*(i+1)*d)%qq] - x[(astart+(i+1)*d)%qq])%modulus
	return x, y, SUCCEEDED







# Routine: printCayleyTable
# input:  none
# output: none

# This routine will generate a Cayley table; the cells are grouped by coset blocks
# with respect to the subgroup generated by q

def printCayleyTable():
	print()
	for i in range(qq):
		for a in range(mm*qq):
			if( a == 0): print(" --",end="")
			else: print("---",end="")
		for a in range(qq):
			print("--",end="")
		print()
		for j in range(mm):
			for k in range(qq):
				print('| ',end="")
				for ell in range(mm):
					print('{:-2} '.format(((i+j*qq)+(k+ell*qq))%(mm*qq)),end="")
			print('|')
	for a in range(mm*qq):
		if( a == 0): print(" --",end="")
		else: print("---",end="")
	for a in range(qq):
		print("--",end="")
	print()






# Routine: printCayleyTableChain
# input:  x  - the rows for the cells in the first chain
#         y  - the columns for the cells in the first chain
#         d  - the index of the block diagonal for the first chain
#         xx - the rows for the cells in the second chain
#         yy - the columns for the cells in the second chain
#         e  - the index of the block diagonal for the second chain
# output: none

# This routine will generate a visual representation of the transversal in the Cayley
# table. In each cell that is used by the transversal, an at symbol (@) will be printed.

def printCayleyTableChain(x,y,d,xx,yy,e):
	print()
	yshift = [0]*qq
	yyshift = [0]*qq
	for i in range(0,qq):
		yshift[(i*d)%qq] = y[((i-1)*d)%qq]
		yyshift[(i*e)%qq] = yy[((i-1)*e)%qq]
	chain1cells = list(zip(x,y)) + list(zip(x,yshift))
	chain2cells = list(zip(xx,yy)) + list(zip(xx,yyshift))
	
	print('Chain 1 (@@): ',end="")
	print('({:-2},{:-2},{:-2})'.format(chain1cells[0][0],chain1cells[0][1],(chain1cells[0][0]+chain1cells[0][1])%modulus),end="")
	for i in range(1,qq):
		print(', ({:-2},{:-2},{:-2})'.format(chain1cells[i][0],chain1cells[i][1],(chain1cells[i][0]+chain1cells[i][1])%modulus),end="")
	print()
	print('Chain 1 Swap: ',end="")
	print('({:-2},{:-2},{:-2})'.format(chain1cells[qq][0],chain1cells[qq][1],(chain1cells[qq][0]+chain1cells[qq][1])%modulus),end="")
	for i in range(qq+1,2*qq):
		print(', ({:-2},{:-2},{:-2})'.format(chain1cells[i][0],chain1cells[i][1],(chain1cells[i][0]+chain1cells[i][1])%modulus),end="")
	print()
	print()
	print('Chain 2 (**): ',end="")
	print('({:-2},{:-2},{:-2})'.format(chain2cells[0][0],chain2cells[0][1],(chain2cells[0][0]+chain2cells[0][1])%modulus),end="")
	for i in range(1,qq):
		print(', ({:-2},{:-2},{:-2})'.format(chain2cells[i][0],chain2cells[i][1],(chain2cells[i][0]+chain2cells[i][1])%modulus),end="")
	print()
	print('Chain 2 Swap: ',end="")
	print('({:-2},{:-2},{:-2})'.format(chain2cells[qq][0],chain2cells[qq][1],(chain2cells[qq][0]+chain2cells[qq][1])%modulus),end="")
	for i in range(qq+1,2*qq):
		print(', ({:-2},{:-2},{:-2})'.format(chain2cells[i][0],chain2cells[i][1],(chain2cells[i][0]+chain2cells[i][1])%modulus),end="")
	print()
	print()
	for i in range(qq):
		for a in range(mm*qq):
			if( a == 0 ): print(" --",end="")
			else: print("---",end="")
		for a in range(qq):
			print("--",end="")
		print()
		for j in range(mm):
			for k in range(qq):
				print('| ',end="")
				for ell in range(mm):
					if( ((i+j*qq),(k+ell*qq)) in chain1cells ):
						print('@@ ',end="")
					elif( ((i+j*qq),(k+ell*qq)) in chain2cells ):
						print('** ',end="")
					else:
						#print('{:-2} '.format(((i+j*qq)+(k+ell*qq))%(mm*qq)),end="")
						print(' . ',end="")
			print('|')
	for a in range(mm*qq):
		if( a == 0 ): print(" --",end="")
		else: print("---",end="")
	for a in range(qq):
		print("--",end="")
	print()






# Routine: printCayleyTableTransversal
# input: transversal - a list of (row,column) pairs
# output: none

# This routine will generate a visual representation of the transversal in the Cayley
# table. In each cell that is used by the transversal, the symbol will be printed.
# In every other cell, a period (.) will be printed.

def printCayleyTableTransversal(transversal):
	print()
	for i in range(qq):
		for a in range(mm*qq):
			if( a == 0 ): print(" --",end="")
			else: print("---",end="")
		for a in range(qq):
			print("--",end="")
		print()
		for j in range(mm):
			for k in range(qq):
				print('| ',end="")
				for ell in range(mm):
					if( ((i+j*qq),(k+ell*qq)) in transversal ):
						print('{:-2} '.format(((i+j*qq)+(k+ell*qq))%(mm*qq)),end="")
					else:
						print(' . ',end="")
			print('|')
	for a in range(mm*qq):
		if( a == 0 ): print(" --",end="")
		else: print("---",end="")
	for a in range(qq):
		print("--",end="")
	print()






# Routine: checkcases
# input: none
# output: none

# This is the main code block. It will loop over all possible partial transversals in
# Z_n, then execute the routine to find a pair of disjoint chains lead to a completion
# to a transversal. There are several means of output; you can print the Cayley table
# with each partial transversal or full transversal highlighted, and you can also have 
# the transversal given in terms of a permutation corresponding to the column locations 
# given its rows.

def checkcases():
	counter = 0
	successes = 0

	# The cells to be paired with (0,0,0) to form a partial transversal are
	# (i,j,i+j) and (k,l,k+l). We first loop through to find all possible 
	# i, j, k, and l which give that {(0,0,0),(i,j,i+j),(k,l,k+l)} is a 
	# partial transveral of length 3 in Z_n.

	for i in range(modulus):
		if (
			# Make sure c_1 is not in the same row as (0,0,0)
			not ( i == 0 ) 
		    ):
			for j in range(modulus):
				if (
					# Make sure c_1 is not in the same column as (0,0,0)
					# Make sure c_1 is not on the main block diagonal (since (0,0,0) already is)
					# Make sure c_1 does not have symbol 0
					not ( j == 0 ) and
					not ( i%qq == j%qq ) and 
					not ( (i+j)%modulus == 0 )
					):
					for k in range(i+1,modulus):
						if (
							# Make sure c_2 is not in the same row as (0,0,0)
							# Make sure c_2 is not in the same row as c_1
							not ( k == 0 ) and 
							not ( i == k )
						   ):
							for l in range(modulus):
								if (  
									# Make sure c_2 is not in the same column as (0,0,0)
									# Make sure c_2 is not in the same column as c_1
									# Make sure c_2 is not on the main block diagonal (since (0,0,0) already is)
									# Make sure c_2 is not on the same block diagonal as c_1
									# Make sure c_2 does not have same symbol as symbol c_1
									# Make sure c_2 does not have symbol 0
									not ( l == 0 ) and 
									not ( j == l ) and
									not ( k%qq == l%qq ) and 
									not ( (i-j)%qq == (k-l)%qq ) and
									not ( (i+j)%modulus == (k+l)%modulus ) and
									not ( 0 == (k+l)%modulus )
								   ):
									x,y,d,xx,yy,e,result = dualChainCreate(i,j,k,l)
									
									counter = counter + 1
									if( result == SUCCEEDED ):
										successes = successes + 1
										# printCayleyTableChain(x,y,d,xx,yy,e)
										# transversal = buildTransversal(x,y,d,xx,yy,e,mm,qq,validPerms,isFound)
										# printCayleyTableTransversal(transversal)
										#linePrintTransversal(i,j,k,l,transversal,mm,qq)
									if(counter % 10000 == 0 ):
										print(counter, file=sys.stderr)
										print(counter)
	print()
	print('Total number of partial transversals found: {}'.format(counter),file=sys.stderr)
	print('Total number of successes in completing to a transversal: {}'.format(successes), file=sys.stderr)
	print('Total number of failures in completing to a transversal: {}'.format(counter-successes), file=sys.stderr)
	print('Total number of partial transversals found: {}'.format(counter))
	print('Total number of successes in completing to a transversal: {}'.format(successes))
	print('Total number of failures in completing to a transversal: {}'.format(counter-successes))







# Routine: linePrintTransversal
# input: i - the row of cell c_1
#        j - the column of cell c_1
#        k - the row of cell c_2
#        l - the column of cell c_2
#        transversal - a set of (row,column) pairs which is a transversal of C(G) 
#                      using (0,0,0), c_1, and c_2
# output: none

# This routine prints out the transversal found which is a completion of the 
# partial transversal containing (0,0,0), (i,j,i+j), and (k,l,k+l).

def linePrintTransversal(i,j,k,l,transversal):
	print()
	print('The completion of ( 0, 0, 0), ({:-2},{:-2},{:-2}) and ({:-2},{:-2},{:-2}) to a transversal of C(Z_{}):'.format(i,j,(i+j)%modulus,k,l,(k+l)%modulus,modulus))
	print()
	mysorted = sorted(transversal, key=lambda tup: tup[0])
	for i in range(qq):
		print( '({:-2},{:-2},{:-2})'.format(mysorted[mm*i][0],mysorted[mm*i][1],(mysorted[mm*i][0]+mysorted[mm*i][1])%modulus),end="" )
		for j in range(1,mm):
			print( ', ({:-2},{:-2},{:-2})'.format(mysorted[mm*i+j][0],mysorted[mm*i+j][1],(mysorted[mm*i+j][0]+mysorted[mm*i+j][1])%modulus),end="" )
		print()







# Routine: partialCompletionArray

# input:  n - the size of H
# output: validPerms - a list of permutations which correspond to transversals in C(H)
#         isFound - a 4-dimensional list which indexes the permutation in validPerms
#                   that is a completion of a completable partial transversal of length 3 in C(H)

# This routine will loop through all possible permutations of H, and for each permutation,
# identify any partial transversals of length 3 (which include (0,0,0) in the cells) for which 
# the permutation corresponds to a completion, and if no other transversal has been designated,
# will be saved as the completion of the partial transversal.

# This gives a way to, when specifying a partial transversal of length 3, easily locate its completion
# to a transversa of C(H), if it has one.

def partialCompletionArray(n):
	isFound = [[[[0 for w in range(n)] for x in range(n)] for y in range(n)] for z in range(n)]

	validPerms = [0]
	permCount = 0

	for comb in combinations(list(range(1,n)),2):
		for perm in permutations(list(range(1,n)),2):
			isFound[comb[0]][comb[1]][perm[0]][perm[1]] = 0
		
	for perm in permutations(range(n)):
		isUsed = [0] * n
		isValidPerm = 1
		if( perm[0] == 0 ):
			for j in range(n):
				if isUsed[(j+perm[j])%n] == 0:
					isUsed[(j+perm[j])%n] = 1
				else:
					isValidPerm = 0

			if isValidPerm == 1:
				permCount = permCount + 1
				validPerms.append(perm)
				for comb in combinations(range(1,n),2):
					isFound[comb[0]][comb[1]][perm[comb[0]]][perm[comb[1]]] = permCount
	return validPerms, isFound







# Routine: find3Transversal

# input:  a - the row of the first cell to be contained in a transversal of C(H). 
#         b - the column of the first cell to be contained in a transversal of C(H). 
#         c - the row of the second cell to be contained in a transversal of C(H). 
#         d - the column of the second cell to be contained in a transversal of C(H). 
#         validPerms - a list of permutations which correspond to transversals in C(H)
#         isFound - an index for completions of partial transversals to transversals in C(H)
# output: transversal - a list of cell locations corresponding to a transversal of C(H)
#                       containing (0,0,0),(a,b,a+b) and (c,d,c+d).

def find3Transversal(a,b,c,d,validPerms,isFound):
	transversal = []
	if( a < c ):
		small = (a//qq,b//qq)
		large = (c//qq,d//qq)
	else:
		small = (c//qq,d//qq)
		large = (a//qq,b//qq)
	index = isFound[small[0]][large[0]][small[1]][large[1]]
	for i in range(mm):
		transversal.append((i*qq,validPerms[index][i]*qq))
	return transversal







# Routine: find2Transversal

# input:  a - the row of the first cell to be contained in a transversal of C(H). 
#         b - the column of the first cell to be contained in a transversal of C(H). 
#         c - the row of the second cell to be contained in a transversal of C(H). 
#         d - the column of the second cell to be contained in a transversal of C(H). 
# output: transversal - a list of cells corresponding to a transversal of C_{i,i}(H)
#                       containing (a,b,a+b) and (c,d,c+d), where a,b in H_i.

def find2Transversal(a,b,c,d):
	coset = a % qq
	transversal = []
	dr = (c//qq-a//qq)%mm
	dc = (d//qq-b//qq)%mm
	for i in range(mm):
		transversal.append((((a//qq+i*dr)%mm)*qq+coset,((b//qq+i*dc)%mm)*qq+coset))
	return transversal







# Routine: buildTransversal

# input:  x  - the rows for the cells in the first chain
#         y  - the columns for the cells in the first chain
#         d  - the index of the block diagonal for the first chain
#         xx - the rows for the cells in the second chain
#         yy - the columns for the cells in the second chain
#         e  - the index of the block diagonal for the second chain
#         validPerms - a list of permutations which correspond to transversals in C(H)
#         isFound - an index for completions of partial transversals to transversals in C(H)
# output: transversal - a list of cell locations corresponding to a transversal of C(G)

# This routine uses two disjoint chains whose swap passes through cells c_1 and c_2, respectively
# to build a transversal which completes (0,0,0), c_1, and c_2. First, the intersection of both 
# chains with the coset blocks on the main block diagonal are completed to transversals.
# The union of these transversals then has the chains removed and replaced with their swaps,
# giving a transversal which is a completion of (0,0,0), c_1, and c_2.

def buildTransversal(x,y,d,xx,yy,e,validPerms,isFound):
	transversal = find3Transversal(x[0],y[0],xx[0],yy[0],validPerms,isFound)
	for i in range(1,qq):
		transversal = transversal + find2Transversal(x[i],y[i],xx[i],yy[i])
	for i in range(0,qq):
			yshift = [0]*qq
	yshift = [0]*qq
	yyshift = [0]*qq
	for i in range(0,qq):
		yshift[(i*d)%qq] = y[((i-1)*d)%qq]
		yyshift[(i*e)%qq] = yy[((i-1)*e)%qq]
	for i in list(zip(x,y)):
		transversal.remove(i)
	for i in list(zip(xx,yy)):
		transversal.remove(i)
	for i in list(zip(x,yshift)):
		transversal.append(i)
	for i in list(zip(xx,yyshift)):
		transversal.append(i)
	return transversal
	







# Execution code

# The code below will randomly generate a partial transversal of length 3 in C(Z_n) which
# contains (0,0,0) and cells belong ot distinct block diagonals. It then produces the 
# chains necessary for completing it to a transversal, then displays such a transversal.
# After completing this example, the code begins the exhaustive search of finding all
# partial transversals of this type and finding completions for each of them.

good_selection = 0
sample_count = 0

while( good_selection == 0 and sample_count < MAX_RUN ):
	sample_count = sample_count + 1
	a  = random.randint(1,modulus-2)
	aa = random.randint(a+1,modulus-1)
	b  = random.randint(1,modulus-1)
	bb = random.randint(1,modulus-1)
	if( b == bb ): continue
	if( (a  + b ) % modulus == 0 ): continue
	if( (aa + bb) % modulus == 0 ): continue
	if( (a + b - aa - bb) % modulus == 0 ): continue
	if( (a-b) % qq == 0 ): continue
	if( (aa-bb) % qq == 0 ): continue
	if( (aa-bb-a+b) % qq == 0 ): continue
	good_selection = 1

if( good_selection == 1 ):
	print()
	print('Below is a Cayley table of order {:-2}, grouped by the natural cosets of <{}>.'.format(modulus,qq) )
	printCayleyTable()

	validPerms,isFound = partialCompletionArray(mm)

	x,y,d,xx,yy,e,result = dualChainCreate(a,b,aa,bb)

	print()
	print('Below is a random example of finding a pair of disjoint chains.')
	print('In this example, we find two disjoint chains whose swaps contain')
	print('({},{},{}) and ({},{},{}), avoid (0,0,0), and their cells in C(<{}>),'.format(a,b,(a+b)%modulus,aa,bb,(aa+bb)%modulus,qq) )
	print('along with (0,0,0), can be completed to a transversal of C(<{}>).'.format(qq) )
	printCayleyTableChain(x,y,d,xx,yy,e)
	transversal = buildTransversal(x,y,d,xx,yy,e,validPerms,isFound)

	print()
	print('Below is a transversal which may be found from the above disjoint chains.')
	linePrintTransversal(a,b,aa,bb,transversal)
	printCayleyTableTransversal(transversal)

	print()
	print('Now the code is looping through all possible partial transversals of')
	print('length 3 which include (0,0,0) and each cell belongs to a distinct block')
	print('diagonal, and attempting to find a completion to a transversal.')
	print()
	print('A counter indicating the number of partial transversals which have been')
	print('investigated is given. When the exhaustive search concludes, the total')
	print('number of such partial transversals the the number of which were')
	print('successfully completed to a transversal in C(Z_{}) are given.'.format(modulus))
	print()
	checkcases()
	print()

