# This code will determine if there are any partial transversals of length 3 in 
# C(G) which are not completable to a transversal in C(G).

# This result is achieved by first finding all transversals in C(G) which
# include (0,0,0), where 0 refers to the zero element of G.
# As each transversal is found, the code identifies the (n-1)(n-2)/2 partial transversals
# of length 3 contained in the transversal which includes (0,0,0).

# After finding all transversals, the code goes through and identifies which partial
# transversals were NOT found in any of the transversals. If all were found, then we
# conclude that each partial transversal of length 3 is completable to a transversal
# in C(G).

# Usage:
#
# > python transversal_check.py Zn 5
# > python transversal_check.py Zn 7
# > python transversal_check.py Zn 9
# > python transversal_check.py ZpxZp 3
#
# The first three runs will classify all partial transversals of Z_5, Z_7, and Z_9
# of length 3 containing (0,0,0) as being completable or non-completable.
# If a partial transversal is completable, a completion is given, first by giving the 
# permutation corresponding to the cell locations, then a permutation associated to the
# symbols of the permutation.
#
# Note, this routine will run for arbitrarily large p and n, but it is not feasible to run
# with an order larger than 9.

import sys
from itertools import permutations
from itertools import combinations

isCyclic = 0

if(len(sys.argv) > 1):
    if(sys.argv[1] == "Zn"):
        n = int(sys.argv[2])
        groupName = "Z_" + str(n)
        isCyclic = 1
    elif(sys.argv[1] == "ZpxZp"):
        p = int(sys.argv[2])
        n = p * p
        groupName = "Z_" + str(p) + " X Z_" + str(p)
    else:
        p = 3
        n = p * p
        groupName = "Z_3 X Z_3"
else:
        p = 3
        n = p * p
        groupName = "Z_3 X Z_3"

# This next routine defines addition in the group. If the group is cyclic, then 
# the operation is simply addition modulo n.
#
# We still use the integers 0,1,...,n-1 to represent the elements of a noncyclic group,
# and we appropriately redefine addition. To perform addition in a noncyclic group, we 
# first map (a,b) to p*a+b. Using this convention, addition must be redefined. For 
# example, (1,1) + (0,p-1) = (1,0) in Z_p x Z_p, so with our identification, we must
# have that (p+1) + (p-1) = p.

# We codify this addition with the routine below. The assumption is that the inputs 
# are integers from 0 to n - 1 to begin with.

def add( a, b ):
    if( isCyclic ):
        return (a + b) % n
    else:
        return p*( ( (a // p) + (b // p) ) % p ) + ( (a + b) % p )

# This code generates the Cayley table for G.

print()
print("The Cayley Table for {}".format(groupName))
print()

print("     ",end="")
for i in range(n):
    print( '{:-3}'.format(i),end="")
print()
print("    -",end="")
for i in range(n):
    print( '---',end="")
print()

for i in range(n):
    print( '{:-3} |'.format(i),end="")
    for j in range(n):
        print('{:-3}'.format(add(i,j)),end="")
    print()

# We catalog the partial transversals using the array below, which takes in four arguments, corresponding to the 
# rows of both cells, then the columns of both cells. For example, the partial transversal
# {(0,0,0), (2,4,6), (3,1,4)} would correspond to isFound[2][3][4][1]. The cells are always ordered 
# in increasing row index to avoid ambiguities.

isFound = [[[[0 for w in range(n)] for x in range(n)] for y in range(n)] for z in range(n)]

# validPerms will keep a list of all permutations phi such that { (a,phi(a),a+phi(a)): a in G } is
# a transversal, and permCount will enumerate them.

validPerms = []
permCount = 0

# We initialize the array keeping track of whether a partial transversal is found or not.
# Note the array isFound is much larger than the number of partial transversals of length 3.
# There is a lot of junk data in the array, but it is easier to write and read the code by sacrificing 
# efficiency in storage space.

for comb in combinations(list(range(1,n)),2):
    for perm in permutations(list(range(1,n)),2):
        isFound[comb[0]][comb[1]][perm[0]][perm[1]] = 0

# This routine determines whether a permutation corresponds to a transversal in C(G)
# When it concludes, validPerms contains all such permutations and there are permCount of them.
# For each valid permutation, we determine which partial tranversals it contains and update isFound.

for perm in permutations(range(n)):

    # isUsed will keep track if the permutation `uses' symbol i in some cell corresponding to perm.
    isUsed = [0] * n
    
    # Again, we sacrifice efficiency for simplicity of reading the code. We only want permutations
    # which fix 0 (to give that (0,0,0) is part of each transversal), so we must check that first.
    if( perm[0] == 0 ):
    
        # This routine will determine if perm is a permutation corresponding to a transversal
        # in C(G); we call such a permutation `valid.' This for loop checks to see if 
        # each symbol in {0,1,...,n-1} is used at most once, which guarantees that the permutation
        # does correspond to a transversal of C(G).

        isValidPerm = 1
        for j in range(n):
            if isUsed[ add( j, perm[j] ) ] == 0:
                isUsed[ add( j, perm[j] ) ] = 1
            else:
                isValidPerm = 0

        # If the permutation does correspond to a transversal of C(G), then we add it to our
        # list of valid permutations and increment the counter. Next, we look at each partial transversal
        # of length 3 contained in the transversal which contains (0,0,0). This is achieved by choosing each
        # pair of cells from the latter n-1 cells in the transversal, then toggling their entry in isFound to 1.
        # In fact, the entry in the isFound array is given a positive integer corresponding to the permutation
        # in validPerms in which the partial transversal is contained. Note there is an index shift, so if 
        # isFound = 5, that means validPerms[4] is a transversal which is a completion.

        if isValidPerm == 1:
            permCount = permCount + 1
            validPerms.append(perm)
            for comb in combinations(range(1,n),2):
                isFound[comb[0]][comb[1]][perm[comb[0]]][perm[comb[1]]] = permCount

# At this point, we need to retrieve which partial transversals are completable and which ones are not.
# This loop runs through all possible cells of C(G) which are in different rows and columns.
# The code first checks to see the corresponding symbols are all distinct.
# Then is looks to the isFound array to see if ... it was found. If not, this code will print the offending
# partial transversal; for example, 0******86 would correspond to {(0,0,0), (7,8,*), (8,6,*)}
# the variable `counter' keeps track of the number of non-completable partial transversals.

print()
print("Incompletable partial transversals" )
print("----------------------------------" )
print()
print("Each partial transversal of length 3 which is incompletable is given below.")
print("For example, 0*2***8** corresponds to the partial transversal with cells" )
print("at (0,0), (2,2), and (6,8).")
print()
counter = 0
for comb in combinations(range(1,n),2):
    for perm in permutations(range(1,n),2):
        if( ( add( comb[0], perm[0] ) == 0 ) or 
            ( add( comb[1], perm[1] ) == 0 ) or 
            ( add( comb[0], perm[0] ) == add( comb[1], perm[1] ) ) ):
            continue
        if( isFound[comb[0]][comb[1]][perm[0]][perm[1]] == 0 ):
            counter = counter + 1
            print('0',end="")
            for i in range(1,n):
                if( comb[0] == i ):
                    print(perm[0],end="")
                elif( comb[1] == i ):
                    print(perm[1],end="")
                else:
                    print('*',end="")
            print()

print()
print('Number of incompletable partial transversals including (0,0,0): {}'.format(counter))

# This code does the same thing as the previous code, except prints out the partial transversals and their completions.
# The first string in the printout is the partial transversal, in terms of its locations, as done in the previous 
# routine. 
# The second string is the permutation corresponding to a transversal which contains the partial transversal. Since
# each symbol is distinct, that means the cells in the transversal all belong to different rows and columns.
# The third string is the set of symbols in the transversal, obtained by adding i and phi(i) in the group. If these are 
# also distinct, this confirms that our transversal really is a transversal.

print()
print()
print("Completions of completable partial transversals" )
print("-----------------------------------------------" )
print()
print("Each partial transversal of length 3 which is completable is given below.")
print("For example, 0*2***8** corresponds to the partial transversal with cells" )
print("at (0,0), (2,2), and (6,8).")
print()
print("After each partial transversal should be two permutations. The first")
print("correponds to the cell locations for a completion of the partial transversal.")
print("The second contains the symbols in each cell of the completion.")
print()
counter = 0
for comb in combinations(range(1,n),2):
    for perm in permutations(range(1,n),2):
        if( ( add( comb[0], perm[0] ) == 0 ) or 
            ( add( comb[1], perm[1] ) == 0 ) or 
            ( add( comb[0], perm[0] ) == add( comb[1], perm[1] ) ) ):
            continue
        if( isFound[comb[0]][comb[1]][perm[0]][perm[1]] > 0 ):
            counter = counter + 1
            print('0',end="")
            for i in range(1,n):
                if( comb[0] == i ):
                    print(perm[0],end="")
                elif( comb[1] == i ):
                    print(perm[1],end="")
                else:
                    print('*',end="")
            print(" --- ",end="")
            
            for i in range(n):
                print(validPerms[isFound[comb[0]][comb[1]][perm[0]][perm[1]]-1][i],end="")

            print(" --- ",end="")
            
            for i in range(n):
                print(add(validPerms[isFound[comb[0]][comb[1]][perm[0]][perm[1]]-1][i],i),end="")
            print()

print()